function Iecg = imecgraphwin(I, bins, ws, s)
% IMECGRAPHWIN EC-Graph feature calculation in a sliding window, for a single-channel image.
%   IECG = IMECGRAPHWIN(I,BINS, WS, S) calculates the EC-Graph vectors for image I of size h x w 
%   using the threshold values in BINS, in a sliding window of size WSxWS, with a step of S
%   (default = 1) pixels.
%
%   The EC-Graph vector for window at location (upper-left corner) i,j (for s=1) is output as
%   row sub2ind([h-ws+1,w-ws+1],i,j) of IECG. 
%   i.e. IECG(sub2ind([h-ws+1,w-ws+1],i,j),:) corresponds to EC(I(i:i+ws-1,j:j+ws-1)>=bins(i)).
%   See demoECGWin.m for a usage demo.
% 
% For questions, please contact eitan.richardson@gmail.com
%
% When using this code, please refer to:
%   Richardson, Eitan, and Michael Werman. 
%   "Efficient classification using the Euler characteristic." 
%   Pattern Recognition Letters 49 (2014): 99-106.
%   http://www.sciencedirect.com/science/article/pii/S0167865514002050

if(~exist('s','var') || isempty(s))
    s = 1;
end

[n,m]=size(I);
l=length(bins);
Iecg=zeros(n-ws+1,m-ws+1,l);
for i=1:l
    Ib=double(I>=bins(i)); % Threshold the image
    IIecg=integralImEcBinary(Ib); % Calculate the integral EC
    Iecg(:,:,i) = IIecg.A(ws+1:end,ws+1:end) - IIecg.B(1:end-ws,ws+1:end) - ...
        IIecg.D(ws+1:end,1:end-ws) + IIecg.C(1:end-ws,1:end-ws);

end

Iecg=Iecg(1:s:end,1:s:end,:);
Iecg=shiftdim(Iecg,2);
Iecg=reshape(Iecg,size(Iecg,1),size(Iecg,2)*size(Iecg,3));
Iecg=Iecg';
end

%% Calculate integral EC of a binary image
function IEC=integralImEcBinary(I)
[n,m]=size(I);

%% Build the integral images of faces, edges and vertices
% Faces
If = cumsum(cumsum(I),2);

% Create a zero-padded version
Ipad=zeros(n+2,m+2);
Ipad(2:end-1,2:end-1)=I;
clear('I');

% 2D integral of vertical edges
Iev=max(Ipad(2:end-1,1:end-1),Ipad(2:end-1,2:end));
Iev=cumsum(cumsum(Iev),2);

% 2D integral of horizontal edges
Ieh=max(Ipad(1:end-1,2:end-1),Ipad(2:end,2:end-1));
Ieh=cumsum(cumsum(Ieh),2);

% 2D integral of vertices
Iv=max(Ipad(:,1:end-1),Ipad(:,2:end));
Iv=max(Iv(1:end-1,:),Iv(2:end,:));
Iv=cumsum(cumsum(Iv),2);

%% Build the four EC integrals

ECa=zeros(n+1,m+1);
ECb=zeros(n+1,m+1);
ECc=zeros(n+1,m+1);
ECd=zeros(n+1,m+1);

% Closed
ECa(:,:) =                              Iv;
ECa(2:end,:) =      ECa(2:end,:) -      Iev;
ECa(:,2:end) =      ECa(:,2:end) -      Ieh;
ECa(2:end,2:end)=   ECa(2:end,2:end) +  If;
% Open on top
ECb(2:end,:) =                          Iv(1:end-1,:);
ECb(2:end,:) =      ECb(2:end,:) -      Iev;
ECb(2:end,2:end) =  ECb(2:end,2:end) -  Ieh(1:end-1,:);
ECb(2:end,2:end) =  ECb(2:end,2:end) +  If;
% Open on top and left
ECc(2:end,2:end) =                      Iv(1:end-1,1:end-1);
ECc(2:end,2:end) =  ECc(2:end,2:end) -  Iev(:,1:end-1);
ECc(2:end,2:end) =  ECc(2:end,2:end) -  Ieh(1:end-1,:);
ECc(2:end,2:end) =  ECc(2:end,2:end) +  If;
% Open on left
ECd(:,2:end) =                          Iv(:,1:end-1);
ECd(2:end,2:end) =  ECd(2:end,2:end) -  Iev(:,1:end-1);
ECd(:,2:end) =      ECd(:,2:end)     -  Ieh;
ECd(2:end,2:end) =  ECd(2:end,2:end) +  If;

IEC.A = ECa;
IEC.B = ECb;
IEC.C = ECc;
IEC.D = ECd;

end

