function ecg=imecgraph(I,bins)
% IMECGRAPH EC-Graph feature calculation for a single-channel image.
%   ECG = IMECGRAPH(I,BINS) calculates the EC-Graph vector for image I using the
%   threshold values in BINS.
%   ECG(i) corresponds to EC(I>=bins(i)).
%   See demo.m for a usage demo.
% 
% For questions, please contact eitan.richardson@gmail.com
%
% When using this code, please refer to:
%   Richardson, Eitan, and Michael Werman. 
%   "Efficient classification using the Euler characteristic." 
%   Pattern Recognition Letters 49 (2014): 99-106.
%   http://www.sciencedirect.com/science/article/pii/S0167865514002050

assert(isa(I,'double'),'Image I must be of type double. Use im2double to convert');
[n,m]=size(I);

% Function over faces (pixels)
gf=I(:);

% Create a zero-padded version
Ipad=zeros(n+2,m+2);
Ipad(2:end-1,2:end-1)=I;

% Calculate function over edges = max function over containing faces
Ie1=max(Ipad(1:end-1,2:end-1),Ipad(2:end,2:end-1));
Ie2=max(Ipad(2:end-1,1:end-1),Ipad(2:end-1,2:end));
ge=[Ie1(:);Ie2(:)];

% Calculate function over vertices = max function over containing faces
Iv=max(Ipad(:,1:end-1),Ipad(:,2:end));
Iv=max(Iv(1:end-1,:),Iv(2:end,:));
gv=Iv(:);

if(~isinf(bins(end)))
    bins(end+1)=Inf;
end

% Count accumulative histograms
hf=histc(gf,bins); hf=cumsum(hf(end-1:-1:1)); hf=hf(end:-1:1);
he=histc(ge,bins); he=cumsum(he(end-1:-1:1)); he=he(end:-1:1);
hv=histc(gv,bins); hv=cumsum(hv(end-1:-1:1)); hv=hv(end:-1:1);

% Calculate Euler Characteritics
ecg=hv-he+hf;
end
