% EC-Graph feature calculation in a sliding window.
% help imecgraphwin for more info.
%
% When using this code, please refer to:
%   Richardson, Eitan, and Michael Werman. 
%   "Efficient classification using the Euler characteristic." 
%   Pattern Recognition Letters 49 (2014): 99-106.
%   http://www.sciencedirect.com/science/article/pii/S0167865514002050
close all ; clear;

N = 16; % Number of threshold values

% Select a reference object window from the first image
I = im2double(imread('AT3_1m4_01.tif'));
I = imadjust(I,stretchlim(I),[]);
I = imresize(I,0.5);
Iref = I(108:150,120:162);
ws = size(Iref,1);
figure,imshow(Iref); title('Reference object window'); movegui('northwest');

% Load a second image (test image)
I = im2double(imread('AT3_1m4_04.tif'));
I = imadjust(I,stretchlim(I),[]);
I = imresize(I,0.5);
[h,w]=size(I);
figure,imshow(I); title('Test image'); movegui('north');

% Define a N-bin threshold edges (last edge will be set to Inf)
T = 0:1/N:0.999;

%% Calculate the referece EC-Graph descriptor vector and plot it
ecgRef = imecgraph(Iref,T);

figure,plot(T,ecgRef,'.-'); 
xlabel('Threshold Value'); ylabel('Euler Characteristic'); 
title('EC-Graph descriptor of the reference object'); grid on;
movegui('northeast'); movegui('northeast');

%% Calculate the EC-Graph in a sliding window for the test image
tic;
ecgResp = imecgraphwin(I,T,ws);
et=toc;
fprintf('EC-Graph calculation of length %d for %d %dx%d windows took %f sec.\n',...
    length(ecgRef), size(ecgResp,1), ws, ws, et);

%% 
% Show the response map
err = sum(abs(ecgResp-repmat(ecgRef',[size(ecgResp,1),1])),2);
Ierr = reshape(exp(-err/100),h-ws+1,w-ws+1);
figure,imagesc(Ierr); title('Detection response'); movegui('southeast');

% Show the highest scoring detection window
[~,mind]=min(err);
[r,c]=ind2sub(size(Ierr),mind);
figure,imshow(I); hold on; title('Highest score detection window'); movegui('southwest');
rectangle('position',[c r ws ws],'edgecolor','r');
