% EC-Graph feature calculation for a triangular mesh object.
% help meshecgraph for more info.
%
% When using this code, please refer to:
%   Richardson, Eitan, and Michael Werman. 
%   "Efficient classification using the Euler characteristic." 
%   Pattern Recognition Letters 49 (2014): 99-106.
%   http://www.sciencedirect.com/science/article/pii/S0167865514002050

clear; close all;

N = 64; % Number of bins

% Create a sphere triange-mech object
[x,y,z] = sphere(100);
fvc=surf2patch(x,y,z,'triangle');

% Create a function over the faces (z+noise)
fvc.facevertexcdata = fvc.vertices(fvc.faces(:,1),3)+randn(size(fvc.faces,1),1)/2;
fvc.facevertexcdata = (fvc.facevertexcdata-min(fvc.facevertexcdata))/(max(fvc.facevertexcdata) - min(fvc.facevertexcdata));
figure, movegui('northwest');
patch(fvc);
shading flat;
view(3);
title(sprintf('A function over a mesh object of %d faces',length(fvc.facevertexcdata)));

% The threshold values
T = 0:1/N:0.999;

tic;
%% Calculatet he EC-Graph descriptor vector
ecg = meshecgraph(fvc.faces,fvc.facevertexcdata,T);

%%
et=toc;
fprintf('EC-Graph calculation of %d values for a %d faces mesh object took %f sec.\n',...
    length(ecg), length(fvc.facevertexcdata),et);

% Plot the result
figure,plot(T,ecg,'.-'); movegui('northeast');
xlabel('Threshold Value'); ylabel('Euler Characteristic'); 
title('EC-Graph Descriptor'); grid on;

% Show the thresholding result
fvc2=fvc;
figure; movegui('south');
for i=1:length(T)
    fvc2.facevertexcdata = double(fvc.facevertexcdata >= T(i));
    patch(fvc2);
    shading flat;
    view(3);
    title(sprintf('t=%0.3f, EC=%d',T(i),ecg(i)));
    pause(0.1);
end

