/**
 * @file
 * Bootstrap Modals.
 */
(function ($, Drupal, Bootstrap) {
  "use strict";

  /**
   * Extend the Bootstrap Modal plugin constructor class.
   */
  Bootstrap.extendPlugin('modal', function (settings) {
    return {
      DEFAULTS: {
        animation: !!settings.modal_animation,
        backdrop: settings.modal_backdrop === 'static' ? 'static' : !!settings.modal_backdrop,
        keyboard: !!settings.modal_keyboard,
        show: !!settings.modal_show,
        size: settings.modal_size
      }
    };
  });

  /**
   * Replace the Bootstrap Modal jQuery plugin definition.
   *
   * Replacing this is needed so that the "option" method can return values.
   */
  Bootstrap.replacePlugin('modal', function () {
    var Modal = this;

    // Extract the arguments.
    var args = Array.prototype.slice.call(arguments, 1);

    // Modal jQuery Plugin Definition.
    return function (option, _relatedTarget) {
      var ret = void(0);
      this.each(function () {
        var $this   = $(this);
        var data    = $this.data('bs.modal');
        var options = $.extend({}, Modal.DEFAULTS, $this.data(), typeof option == 'object' && option);

        if (!data) $this.data('bs.modal', (data = new Modal(this, options)));
        if (typeof option == 'string') ret = data[option].apply(data, args);
        else if (options.show) data.show(_relatedTarget);
      });

      // If just one element and there was a result returned for the option passed,
      // then return the result. Otherwise, just return the jQuery object.
      return this.length === 1 && ret !== void(0) ? ret : this;
    }
  });

  /**
   * Extend Drupal theming functions.
   */
  $.extend(Drupal.theme, /** @lend Drupal.theme */ {
    /**
     * Theme function for a Bootstrap Modal.
     *
     * @param {object}[variables]
     *   An object with the following keys:
     *   - title: The name of the tab.
     *
     * @return {string}
     *   The HTML for the modal.
     */
    bootstrapModal: function (variables) {
      var settings = drupalSettings.bootstrap || {};
      var defaults = {
        body: '',
        closeButton: true,
        description: {
          content: null,
          position: 'before'
        },
        footer: '',
        id: 'drupal-modal',
        size: settings.modal_size ? settings.modal_size : '',
        title: Drupal.t('Loading...')
      };
      variables = $.extend(true, {}, defaults, variables);
      var output = '';

      // Build the modal wrapper.
      var classes = ['modal'];
      if (settings.modal_animation) {
        classes.push('fade');
      }
      output += '<div id="' + variables.id + '" class="' + classes.join(' ') + '" tabindex="-1" role="dialog">';

      // Build the modal-dialog wrapper.
      var dialogClasses = ['modal-dialog'];
      if (variables.size) {
        // @todo This should really be a clean CSS class method instead.
        dialogClasses.push(Drupal.checkPlain(variables.size));
      }
      output += '<div class="' + dialogClasses.join(' ') + '" role="document">';

      // Build the modal-content wrapper.
      output += '<div class="modal-content">';

      // Build the header wrapper and title.
      output += Drupal.theme.bootstrapModalHeader(variables.title, variables.closeButton);

      // Build the body.
      output += Drupal.theme.bootstrapModalBody(variables.id + '--body', variables.body, variables.description);

      // Build the footer.
      output += Drupal.theme.bootstrapModalFooter(variables.footer);

      // Close the modal-content wrapper.
      output += '</div>';

      // Close the modal-dialog wrapper.
      output += '</div>';

      // Close the modal wrapper.
      output += '</div>';

      // Return the constructed modal.
      return output;
    },

    /**
     * Theme function for a Bootstrap Modal body markup.
     *
     * @param {string} id
     *   A unique ID for the modal body div.
     * @param {string} body
     *   The HTML markup to place in the body.
     * @param {string|object} description
     *   A description to show. Can either be a string or an object with the
     *   following key/value pairs:
     *   - content: The description value.
     *   - position: (optional) A display setting that can have these values:
     *     - before: The description is displayed before the body. This is the
     *       default value.
     *     - after: The description is display after the body.
     *     - invisible: The description is displayed after the element, hidden
     *       visually but available to screen readers.
     *
     * @return {string}
     *   The HTML for the modal close button.
     */
    bootstrapModalBody: function (id, body, description) {
      var output = '';
      output += '<div id="' + id + '" class="modal-body">';
      if (!description || !$.isPlainObject(description)) {
        description = { content: description};
      }
      description = $.extend({ position: 'before' }, description);

      var descriptionClasses = ['help-block'];
      if (description.content && description.position === 'invisible') {
        descriptionClasses.push('sr-only');
      }
      if (description.content && description.position === 'before') {
        output += '<p class="' + descriptionClasses.join(' ') + '">' + description.content + '</p>';
      }
      output += body;
      if (description.content && (description.position === 'after' || description.position === 'invisible')) {
        output += '<p class="' + descriptionClasses.join(' ') + '">' + description.content + '</p>';
      }
      output += '</div>';
      return output;
    },

    /**
     * Theme function for a Bootstrap Modal close button.
     *
     * @return {string}
     *   The HTML for the modal close button.
     */
    bootstrapModalClose: function () {
      return '<button type="button" class="close" data-dismiss="modal" aria-label="' + Drupal.t('Close') + '"><span aria-hidden="true">&times;</span></button>';
    },

    /**
     * Theme function for a Bootstrap Modal footer.
     *
     * @param {string} [footer]
     *   The HTML markup to place in the footer.
     * @param {boolean} [force]
     *   Flag to force the rendering of the footer.
     *
     * @return {string}
     *   The HTML for the modal footer.
     */
    bootstrapModalFooter: function (footer, force) {
      return footer || force ? '<div class="modal-footer">' + (footer || '') + '</div>' : '';
    },

    /**
     * Theme function for a Bootstrap Modal header.
     *
     * @param {string} [title]
     *   The title for the header.
     * @param {boolean} [closeButton]
     *   Flag indicating whether or not to show the close button in the header.
     *
     * @return {string}
     *   The HTML for the modal header.
     */
    bootstrapModalHeader: function (title, closeButton) {
      var output = '';
      if (title) {
        closeButton = closeButton !== void(0) ? closeButton : true;
        output += '<div class="modal-header">';
        if (closeButton) {
          output += Drupal.theme.bootstrapModalClose();
        }
        output += '<h4 class="modal-title">' + Drupal.checkPlain(title) + '</h4>';
        output += '</div>';
      }
      return output;
    }
  })

})(window.jQuery, window.Drupal, window.Drupal.bootstrap);
